<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AirportDetail;
use App\Models\SearchQuery;
use App\Models\HotalQuery;
use App\Models\GroupTicketQuery;
use App\Models\VisaQuery;
use App\Models\TourModel;
use App\Models\UmrahEnquiry;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Mail;

class SiteController extends Controller
{
    public function homaPage(){
        $allAirports = AirportDetail::all();
        return view('index', compact('allAirports'));
    }
    public function allTours(){
        return view('alltours');
    }
    public function thailandTour(){
        return view('tours.thailandtour');
    }
    public function bakuAndIstanbulTour(){
        return view('tours.bakuistanbul');
    }
    public function srilankaTour(){
        return view('tours.srilanka');
    }
    public function countriesWithTomatinaTour(){
        return view('tours.tomatina');
    }
    public function submitSearch(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'from' => 'required|string|max:255',
            'to' => 'required|string|max:255',
            'depart_date' => 'required|date',
            'return_date' => 'nullable|date',
            'direct-flight-only' => 'sometimes|in:on',
            'js-count-adult' => 'required|integer',
            'js-count-child' => 'required|integer',
            'js-count-infants' => 'required|integer',
            'flight-type' => 'required|string|max:255',
        ]);

        // Store the validated data in the database
        $searchQuery = new SearchQuery();
        $searchQuery->flying_from = $validated['from'];
        $searchQuery->flying_to = $validated['to'];
        $searchQuery->depart_date = $validated['depart_date'];
        $searchQuery->return_date = $validated['return_date'] ?? null;
        $searchQuery->direct_flight_only = isset($validated['direct-flight-only']) ? 'yes' : 'no';
        $searchQuery->adult_count = $validated['js-count-adult'];
        $searchQuery->child_count = $validated['js-count-child'];
        $searchQuery->infant_count = $validated['js-count-infants'];
        $searchQuery->flight_type = $validated['flight-type'];
        $searchQuery->save();

        // Redirect to the contact form with the search query ID
        return Redirect::route('query-contact', ['id' => $searchQuery->id]);
    }
    public function submitHotal(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:15',
            'want_to_stay' => 'required|string|max:255',
            'destination' => 'required|string|max:255',
            'room_type' => 'required|string|max:255',
            'check_in' => 'required|date',
            'check_out' => 'required|date|after_or_equal:check_in',
            'message' => 'nullable|string|max:1000',
        ]);

        // Store the validated data in the database
        $hotalQuery = new HotalQuery();
        $hotalQuery->name = $validated['name'];
        $hotalQuery->email = $validated['email'];
        $hotalQuery->phone = $validated['phone'];
        $hotalQuery->want_to_stay = $validated['want_to_stay'];
        $hotalQuery->destination = $validated['destination'];
        $hotalQuery->room_type = $validated['room_type'];
        $hotalQuery->check_in = $validated['check_in'];
        $hotalQuery->check_out = $validated['check_out'];
        $hotalQuery->message = $validated['message'] ?? null;
        $hotalQuery->save();

        // Send mail
        Mail::send('emails.hotal-query', ['data' => $validated], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Hotel Booking Query Submission');
        });

        // Return a response (e.g., redirect or JSON response)
        return Redirect::route('thank-you')->with('success', 'Your hotel query has been submitted successfully.');
    }
    public function submitVisa(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:15',
            'country_for_visa' => 'required|string|max:255',
            'visa_type' => 'required|string|max:50',
            'message' => 'nullable|string|max:1000',
        ]);

        // Store the validated data in the database
        $visaQuery = new VisaQuery();
        $visaQuery->name = $validated['name'];
        $visaQuery->email = $validated['email'];
        $visaQuery->phone = $validated['phone'];
        $visaQuery->country_for_visa = $validated['country_for_visa'];
        $visaQuery->visa_type = $validated['visa_type'];
        $visaQuery->message = $validated['message'] ?? null;
        $visaQuery->save();

        // Send mail
        Mail::send('emails.visa-query', ['data' => $validated], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Visa Query Submission');
        });

        // Return a response (e.g., redirect or JSON response)
        return Redirect::route('thank-you')->with('success', 'Your query has been submitted successfully.');
    }

    public function submitGroupTicket(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:15',
            'destination' => 'required|string|max:255',
            'depart_from' => 'required|date',
            'depart_to' => 'required|date|after_or_equal:depart_from',
            'js-count-adult' => 'required|integer|min:1',
            'js-count-child' => 'nullable|integer|min:0',
            'js-count-infants' => 'nullable|integer|min:0',
            'message' => 'nullable|string|max:1000',
        ]);

        // Store the validated data in the database
        $groupTicketQuery = new GroupTicketQuery();
        $groupTicketQuery->name = $validated['name'];
        $groupTicketQuery->email = $validated['email'];
        $groupTicketQuery->phone = $validated['phone'];
        $groupTicketQuery->destination = $validated['destination'];
        $groupTicketQuery->depart_from = $validated['depart_from'];
        $groupTicketQuery->depart_to = $validated['depart_to'];
        $groupTicketQuery->adults = $validated['js-count-adult'];
        $groupTicketQuery->children = $validated['js-count-child'] ?? 0;
        $groupTicketQuery->infants = $validated['js-count-infants'] ?? 0;
        $groupTicketQuery->message = $validated['message'] ?? null;
        $groupTicketQuery->save();

        // Send mail
        Mail::send('emails.group-ticket-query', ['data' => $validated], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Group Ticket Query Submission');
        });

        // Return a response (e.g., redirect or JSON response)
        return Redirect::route('thank-you')->with('success', 'Your group ticket query has been submitted successfully.');
    }

    public function submitUmrahEnquiry(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'lead_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:15',
            'departure_city' => 'required|string|max:255',
            'departure_date' => 'required|date',
            'night_in_madinah' => 'required|string|max:255',
            'js-count-adult' => 'required|integer|min:1',
            'js-count-child' => 'nullable|integer|min:0',
            'js-count-infants' => 'nullable|integer|min:0',
            'message' => 'nullable|string|max:1000',
        ]);

        // Store the validated data in the database
        $umrahEnquiry = new UmrahEnquiry();
        $umrahEnquiry->lead_name = $validated['lead_name'];
        $umrahEnquiry->email = $validated['email'];
        $umrahEnquiry->phone = $validated['phone'];
        $umrahEnquiry->departure_city = $validated['departure_city'];
        $umrahEnquiry->departure_date = $validated['departure_date'];
        $umrahEnquiry->night_in_madinah = $validated['night_in_madinah'];
        $umrahEnquiry->adults = $validated['js-count-adult'];
        $umrahEnquiry->children = $validated['js-count-child'] ?? 0;
        $umrahEnquiry->infants = $validated['js-count-infants'] ?? 0;
        $umrahEnquiry->message = $validated['message'] ?? null;
        $umrahEnquiry->save();

        // Send mail
        Mail::send('emails.umrah-enquiry', ['data' => $validated], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Umrah Enquiry Submission');
        });

        // Return a response (e.g., redirect or JSON response)
        return Redirect::route('thank-you')->with('success', 'Your Umrah enquiry has been submitted successfully.');
    }

    public function submitTour(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:15',
            'no_of_traveler' => 'required|integer|min:1',
            'selected_tour' => 'required|string|max:255',
            'depart_date' => 'required|date|after:today',
            'message' => 'nullable|string|max:1000',
        ]);

        // Store the validated data in the database
        $tourQuery = new TourModel();
        $tourQuery->name = $validated['name'];
        $tourQuery->email = $validated['email'];
        $tourQuery->phone = $validated['phone'];
        $tourQuery->no_of_traveler = $validated['no_of_traveler'];
        $tourQuery->selected_tour = $validated['selected_tour'];
        $tourQuery->depart_date = $validated['depart_date'];
        $tourQuery->message = $validated['message'] ?? null;
        $tourQuery->save();

        // Send mail
        Mail::send('emails.tour-query', ['data' => $validated], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Tour Query Submission');
        });

        // Return a response (e.g., redirect or JSON response)
        return Redirect::route('thank-you')->with('success', 'Your query has been submitted successfully.');
    }
    public function queryContact($id)
    {
        return view('querycontact', ['id' => $id]);
    }
    public function submitQueryContact(Request $request)
    {
        // Find the search query by ID
        $searchQuery = SearchQuery::find($request->id);

        if (!$searchQuery) {
            return redirect()->route('home')->with('error', 'Search query not found.');
        }

        // Validate the contact form data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        // Update the search query with contact details
        $searchQuery->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'subject' => $validated['subject'],
            'message' => $validated['message'],
        ]);

        // Send a single email with both forms' data
        Mail::send('emails.search-and-contact', ['searchQuery' => $searchQuery], function ($message) use ($validated) {
            $message->to('flyingexperto.pk@gmail.com')
                    ->subject('Your Search and Contact Details');
        });

        // Redirect to a thank-you page
        return Redirect::route('thank-you')->with('success', 'Your query has been submitted successfully.');
    }
    public function thankYou(){
        return view('thankyou');
    }
    public function visaChecklist(){
        return view('visachecklist');
    }
    public function contact(){
        return view('contact');
    }
}
